
import json, csv, hashlib, sys, math, os
from pathlib import Path

ROOT = Path(__file__).resolve().parent

def sha256_file(p: Path) -> str:
    import hashlib
    h = hashlib.sha256()
    with open(p, "rb") as f:
        for chunk in iter(lambda: f.read(65536), b""):
            h.update(chunk)
    return h.hexdigest()

def load_prereg():
    with open(ROOT / "prereg" / "preregister.json","r",encoding="utf-8") as f:
        return json.load(f)

def load_study_tolerances():
    # very small YAML reader for the tolerances block in manifests/study.yaml
    tol = {}
    path = ROOT / "manifests" / "study.yaml"
    current = None
    for line in path.read_text(encoding="utf-8").splitlines():
        s = line.strip()
        if s.startswith("tolerances:"):
            current = "tol"
            continue
        if current == "tol":
            if not s or ":" not in s: 
                continue
            k, v = s.split(":",1)
            k = k.strip()
            v = v.strip()
            try:
                tol[k] = float(v)
            except:
                pass
    return tol

def read_earth_ring_stats():
    p = ROOT / "audits" / "earth_ring.stats.json"
    if p.exists():
        with open(p,"r",encoding="utf-8") as f:
            return json.load(f)
    # else compute from earth_ring.csv
    import statistics as st
    rows = []
    with open(ROOT / "audits" / "earth_ring.csv","r",encoding="utf-8") as f:
        R = csv.DictReader(f)
        for r in R:
            rows.append({"mesh":r["mesh"], "plateau": float(r["plateau"])})
    coarse = [r["plateau"] for r in rows if r["mesh"]=="coarse"]
    fine   = [r["plateau"] for r in rows if r["mesh"]=="fine"]
    def mean(x): return sum(x)/len(x)
    s_coarse, s_fine = mean(coarse), mean(fine)
    pooled = (s_coarse + s_fine)/2.0
    import math
    def cv(x):
        m = mean(x)
        var = sum((a-m)**2 for a in x)/(len(x)-1)
        return math.sqrt(var)/m
    return {"Sring_coarse_mean": s_coarse, "Sring_fine_mean": s_fine, "Sring_pooled": pooled,
            "cv_coarse": cv(coarse), "cv_fine": cv(fine), "mesh_delta": abs(s_coarse-s_fine)/pooled, "pass": True}

def verify_hashes_match_rows(rows):
    # compare per-row hashes to on-disk
    # manifest_hash = sha256(manifests/manifest.yaml)
    man_path = ROOT / "manifests" / "manifest.yaml"
    manifest_hash_disk = sha256_file(man_path)
    # dag_hash is stored inside the YAML as a literal; we just check it matches row values
    dag_hash_yaml = None
    for line in man_path.read_text(encoding="utf-8").splitlines():
        s = line.strip()
        if s.startswith("dag_hash:"):
            # dag_hash: "<hash>"
            try:
                dag_hash_yaml = s.split(":",1)[1].strip().strip('"').strip("'").strip()
            except:
                pass
            break

    cat_hash = (ROOT / "catalog" / "catalog_hash.txt").read_text(encoding="utf-8").strip()
    hinge_hash = (ROOT / "hinges" / "hinge_hash.txt").read_text(encoding="utf-8").strip()

    for r in rows:
        assert r["manifest_hash"] == manifest_hash_disk, f"manifest_hash mismatch in row: {r['manifest_hash']} vs {manifest_hash_disk}"
        assert r["dag_hash"] == dag_hash_yaml, f"dag_hash mismatch in row: {r['dag_hash']} vs {dag_hash_yaml}"
        assert r["catalog_hash"] == cat_hash, f"catalog_hash mismatch in row: {r['catalog_hash']} vs {cat_hash}"
        assert r["hinge_hash"] == hinge_hash, f"hinge_hash mismatch in row: {r['hinge_hash']} vs {hinge_hash}"

def verify_deflection(prereg, tol, er_stats):
    # Load rows
    import pandas as pd
    df = pd.read_csv(ROOT / "runs" / "deflection.csv")
    rows = df.to_dict(orient="records")

    # Hash check
    verify_hashes_match_rows(rows)

    # pooled Sring and R_oplus from CSV
    Sring = er_stats.get("Sring_pooled", None)
    if Sring is None:
        Sring = (er_stats["Sring_coarse_mean"] + er_stats["Sring_fine_mean"]) / 2.0
    R_oplus = float(rows[0]["R_oplus"])

    # Mesh Athetahat
    mesh_A = {}
    for m in ("coarse","fine"):
        sub = [r for r in rows if r["mesh"]==m]
        assert len(sub) > 0, f"no rows for mesh={m}"
        Ahat = float(sub[0]["Atheta_fit"])
        Khat = Ahat / (Sring * R_oplus)
        mesh_A[m] = (Ahat, Khat, float(sub[0]["flat_RMSE"]), float(sub[0]["flat_slope_CI_lo"]), float(sub[0]["flat_slope_CI_hi"]))

    # Flatness
    assert mesh_A["coarse"][2] <= tol["flat_rmse"], f"coarse flatness RMSE {mesh_A['coarse'][2]} > {tol['flat_rmse']}"
    assert mesh_A["fine"][2]   <= tol["flat_rmse"], f"fine flatness RMSE {mesh_A['fine'][2]} > {tol['flat_rmse']}"
    assert mesh_A["coarse"][3] <= 0.0 <= mesh_A["coarse"][4], "coarse flatness slope CI does not include 0"
    assert mesh_A["fine"][3]   <= 0.0 <= mesh_A["fine"][4],   "fine flatness slope CI does not include 0"

    # Ratio
    Klo = float(prereg["ratios"]["K_theta"]["ci_lo"])
    Khi = float(prereg["ratios"]["K_theta"]["ci_hi"])
    assert Klo <= mesh_A["coarse"][1] <= Khi, f"Ktheta_hat coarse {mesh_A['coarse'][1]} outside prereg [{Klo},{Khi}]"
    assert Klo <= mesh_A["fine"][1]   <= Khi, f"Ktheta_hat fine {mesh_A['fine'][1]} outside prereg [{Klo},{Khi}]"

    # Mesh agreement
    A_c, A_f = mesh_A["coarse"][0], mesh_A["fine"][0]
    A_bar = (A_c + A_f)/2.0
    assert abs(A_c - A_f) <= tol["tau_mesh"] * A_bar, f"mesh amplitude delta {(abs(A_c-A_f)/A_bar):.4f} > tau_mesh {tol['tau_mesh']}"

    # Window audits
    mesh_cert = json.load(open(ROOT / "audits" / "mesh_cert.json","r",encoding="utf-8"))
    assert mesh_cert["pass"] and mesh_cert["corr"] >= 0.99 and mesh_cert["amp_delta"] <= tol["tau_mesh"], "mesh-window certification failed"

    # Global audits
    for name in ["neutrality_audit.json","sr_fit.json","isotropy.json","monotone.json","nosig.json","diag_leak.json","rng_ties.json"]:
        a = json.load(open(ROOT / "audits" / name,"r",encoding="utf-8"))
        assert a["pass"] == True, f"audit {name} failed"

    # Claimable flags
    for r in rows:
        assert str(r["claimable"]).lower() == "true", "found non-claimable deflection row"

    return {"Ahat_coarse": mesh_A["coarse"][0], "Ahat_fine": mesh_A["fine"][0], "Khat_coarse": mesh_A["coarse"][1], "Khat_fine": mesh_A["fine"][1]}

def verify_recentering(tol):
    import pandas as pd, numpy as np, json
    dr = pd.read_csv(ROOT / "runs" / "recentering.csv")
    rows = dr.to_dict(orient="records")

    # Sanity: hashes consistent with deflection rows
    verify_hashes_match_rows(rows)

    # Metrics
    slope_rel_errors = []
    ratio_deltas = []
    for r in rows:
        s_target = float(r["s_target"])
        s_hat = float(r["s_hat"])
        slope_rel_errors.append(abs(s_hat - s_target) / s_target)
        ratio_deltas.append(abs(float(r["ratio_invariance_delta"])))

        assert str(r["claimable"]).lower() == "true", "found non-claimable recentering row"

    max_slope_err = max(slope_rel_errors) if slope_rel_errors else 0.0
    max_ratio_delta = max(ratio_deltas) if ratio_deltas else 0.0

    # Check against tolerance
    assert max_slope_err <= tol["tau_rc"], f"recenter slope rel error {max_slope_err:.4f} > tau_rc {tol['tau_rc']}"
    assert max_ratio_delta <= tol["tau_ratio"], f"recenter ratio delta {max_ratio_delta:.4f} > tau_ratio {tol['tau_ratio']}"

    # Audit file
    rc = json.load(open(ROOT / "audits" / "recentering_audit.json","r",encoding="utf-8"))
    assert rc["pass"] == True, "recentering audit failed"

    return {"max_slope_rel_error": max_slope_err, "max_ratio_delta": max_ratio_delta}

def main():
    prereg = load_prereg()
    tol = load_study_tolerances()
    er_stats = read_earth_ring_stats()

    # Earth-ring tolerances
    assert er_stats["cv_coarse"] <= tol["tau_CV"], f"ring CV coarse {er_stats['cv_coarse']:.4f} > {tol['tau_CV']}"
    assert er_stats["cv_fine"]   <= tol["tau_CV"], f"ring CV fine {er_stats['cv_fine']:.4f} > {tol['tau_CV']}"
    assert er_stats["mesh_delta"]<= tol["tau_mesh"], f"ring mesh Δ {er_stats['mesh_delta']:.4f} > {tol['tau_mesh']}"

    d_metrics = verify_deflection(prereg, tol, er_stats)
    r_metrics = verify_recentering(tol)

    out = {"pass": True, "deflection": d_metrics, "recentering": r_metrics, "notes": "All acceptance checks passed."}
    with open(ROOT / "verification_report.json","w",encoding="utf-8") as f:
        json.dump(out, f, indent=2)
    print("✅ Verification PASSED")
    print(json.dumps(out, indent=2))

if __name__ == "__main__":
    try:
        main()
    except AssertionError as e:
        print("❌ Verification FAILED:", str(e))
        sys.exit(2)
    except Exception as e:
        print("❌ Verification ERROR:", repr(e))
        sys.exit(3)
